/* Copyright (C) 2016-2018 RealVNC Ltd.  All Rights Reserved.
 */

#ifndef VNCCOMMONDECODER_H_1665178272128288539943113463580786355
#define VNCCOMMONDECODER_H_1665178272128288539943113463580786355

/**
 * \file vnccommondecoder.h
 *
 * This header file, along with vnccommondecodertypes.h and
 * vnccommondecodersupportingapi.h, defines the interface which should be
 * implemented by Decoder Plugins wishing to integrate with the Common
 * Decoder Framework.
 *
 * The Common Decoder Framework is a generic media decoding framework which
 * will be used by various VNC Automotive SDKs.
 *
 * This particular header defines the Decoder Callbacks, which are function
 * calls implemented by the decoder, and triggered by the framework.
 *
 * \mainpage Common Decoder Framework: Decoder API
 *
 * \note Note that a convenient C++ wrapper, called "BaseDecoder", exists for
 * this API. It can be found inside the distribution package alongside the
 * sample decoders.
 *
 * \section Introduction Introduction
 *
 * The Common Decoder Framework is a generic media decoding framework which
 * will be used by various VNC Automotive SDKs. The Framework loads and manages
 * decoder plugins.
 *
 * The Decoder API provides access to raw streams of media, allowing you to
 * decode and render these streams in a manner which makes full use of your
 * specific hardware platform.
 *
 * \section Definitions Definitions
 *
 * <b>Decoder Plugin</b> A piece of software making use of the Decoder API.
 * This may be a shared library, or implemented directly by the SDK user and
 * registered as a function pointer.
 *
 * <b>Decoder API</b> The combination of the Entry Point, the Decoder Callbacks
 * and the Supporting API.
 *
 * <b>Entry point</b> The initialisation function for the decoder plugin. In
 * shared-library-based decoder plugins, this is exported as
 * \ref VNCCommonDecoderInitializeType "VNCCommonDecoderInitialize". For
 * function-pointer-based decoder plugins, the function pointer which is
 * registered points to the entry point.
 *
 * <b>Decoder Callback</b> A C callback implemented by the decoder plugin, to
 * be invoked by the Framework. The names of these all begin with
 * VNCCommonDecoderCallback.
 *
 * <b>Supporting API</b> The C API exposed by the Common Decoder Framework, to
 * be invoked by the decoder plugin. The names of these all begin with
 * VNCCommonDecoderSupportingAPI.
 *
 * <b>SDK User</b> The application making using of the relevant RealVNC SDK.
 * This may be the Automotive Viewer, or your own custom application.
 *
 * <b>Framework User</b> The software making use of a decoder plugin through
 * the Framework. This will usually be an SDK, for example the CarLife SDK.
 *
 * <b>SDK Thread</b> The main thread of the SDK. While one or more streams are
 * active, all Decoder Callbacks must be invoked from the SDK thread.
 *
 * \section DecoderPlugins Decoder Plugins
 *
 * Decoder plugins can either be loaded by name from dynamic libraries
 * (exporting a function called VNCCommonDecoderInitialize), or an application
 * can register a function pointer directly.
 *
 * The first function which is invoked will be the entry point
 * (VNCCommonDecoderInitialize) after which the decoder will start receiving
 * calls to its registered callback functions.
 *
 * A decoder can request destruction at any time using the
 * \ref VNCCommonDecoderSupportingAPIError supporting API. Calls to this
 * API will be passed up to the framework user, which will shut down the current
 * connection and destroy the decoder. The framework user is not required to
 * take any immediate action though, and further decoder callbacks may be
 * received while the shutdown process is occurring.
 *
 * The decoder may only be destroyed when there are no active streams.
 *
 * \ref VNCCommonDecoderCallbackDestroyed is a synchronous callback. When
 * destroying the decoder, all resources must be released by the time
 * \ref VNCCommonDecoderCallbackDestroyed has returned.
 *
 * The SDK is responsible for ensuring that decoder callbacks are not called
 * simultaneously. While one or more streams are in existence, the SDK must
 * only invoke decoder callbacks from the SDK thread. If no streams exist, then
 * the SDK may invoke decoder callbacks from any thread, but must ensure that
 * invocations are thread safe.
 *
 * \section MediaStreams Media Streams
 *
 * Once a decoder has been created, it can be informed of one or more media
 * streams. The decoder can choose how to decode, mix, and render each stream
 * individually.
 *
 * While it exists, a stream can occupy one of two states: "Stopped", and
 * "Playing". A set of four decoder callbacks are used by the framework to tell
 * the decoder to move a particular stream from one state to another.
 *
 * Creating, starting, stopping, and destroying the stream are asynchronous
 * operations, and supporting APIs will be provided to allow the decoder to
 * notify the framework (and hence, the framework user) when they have
 * completed.
 *
 * As with all other supporting APIs other than
 * \ref VNCCommonDecoderSupportingAPILog, the decoder may invoke these state
 * change supporting API callbacks only from the SDK thread.
 *
 * The decoder may invoke the state change supporting API callbacks from any
 * decoder callback, including the one which triggered the change.
 *
 * The decoder may, at any time while the stream exists, receive
 * \ref VNCCommonDecoderCallbackStreamPayload callbacks containing timestamped
 * payload data. It is up to the decoder to decide what to do with payloads
 * received while the stream is not playing.
 *
 * The framework user may attempt to change the state of the decoder while
 * another state change is pending. For example, it may start the stream before
 * the decoder has finished creating it. The decoder must handle each of these
 * state changes in the correct order.
 *
 * The decoder should not move from the "Playing" state to the "Stopped" state
 * (and hence, give the \ref VNCCommonDecoderSupportingAPIStreamStopCompleted
 * callback) until it has finished rendering all the payloads received before
 * \ref VNCCommonDecoderCallbackStreamStopped was invoked. However, if the
 * stopImmediately flag was set during the
 * \ref VNCCommonDecoderCallbackStreamStopped callback, the decoder need not
 * wait.
 *
 * Additionally, before invoking
 * \ref VNCCommonDecoderSupportingAPIStreamDestroyCompleted, the decoder must
 * have recycled all the payloads it has received for that stream (using
 * \ref VNCCommonDecoderSupportingAPIStreamRecyclePayload).
 *
 * The framework user must not provide any callbacks for a stream after
 * triggering \ref VNCCommonDecoderCallbackStreamDestroyed.
 *
 * The decoder may receive "format hint" and "content hint" callbacks at any
 * time, which contain the framework user’s best guess at kind of data the
 * payloads contain. Although these are allowed at any time, the framework user
 * should attempt to provide these while the decoder is stopped.
 *
 * The decoder may request the termination of a particular stream at any given
 * time, using the \ref VNCCommonDecoderSupportingAPIStreamError
 * supporting API. Calls to this API will be passed up to the framework user,
 * which may decide to shut down the current connection, but will likely keep
 * the decoder running to process streams from the next connection. The
 * framework user is not required to take any immediate action though, and
 * further decoder callbacks may be received while the shutdown process is
 * occurring.
 *
 * Note that, if the decoder is completely unable to continue (e.g. due to an
 * issue with the decoding hardware), the
 * \ref VNCCommonDecoderSupportingAPIError supporting API should be used
 * instead, which is not specific to any particular stream.
 *
 * \section section_legal Legal Information
 *
 * Copyright (C) 2016-2018 RealVNC Ltd. All Rights Reserved.
 *
 * RealVNC and VNC are trademarks of RealVNC Limited and are protected by
 * trademark registrations and/or pending trademark applications in the
 * European Union, United States of America and other jurisdictions.
 *
 * Other trademarks are the property of their respective owners.
 *
 * Protected by UK patents 2481870, 2491657; US patents 8760366, 9137657; EU patent 2652951.
 */

#include <vnccall.h>

#include <cdfdecoder/vnccommondecodertypes.h>
#include <cdfdecoder/vnccommondecodersupportingapi.h>

#ifdef __cplusplus
extern "C"
{
#endif


/** \defgroup decodercallbacks Decoder Callbacks */

/**
 * \brief The last callback invoked on the decoder. No further decoder
 * callbacks should be received once this has returned.
 * \ingroup decodercallbacks
 *
 * This callback is not optional, and must be implemented by the decoder.
 *
 * This should only be invoked once all streams have been destroyed.
 *
 * This is a synchronous callback, and the decoder is considered to be in the
 * "Destroyed" state by the time it has returned.
 *
 * No value is returned – there is no need to report errors from this callback.
 *
 * All resources allocated by the decoder should have been released by the time
 * this callback returns.
 *
 * \param pluginContext The context pointer returned by the entry point.
 */
typedef void VNCCALL
VNCCommonDecoderCallbackDestroyed(
    VNCCommonDecoderPluginContext pluginContext);

/**
 * \brief Informs the decoder that a new stream is available.
 * \ingroup decodercallbacks
 *
 * This callback is not optional, and must be implemented by the decoder.
 *
 * This is an asynchronous callback. The decoder must notify the framework that
 * the stream has been created using the
 * VNCCommonDecoderSupportingAPIStreamCreateCompleted supporting API.
 *
 * The amount of detail provided to the decoder about the stream has
 * deliberately been limited, as the encoding and content can change while the
 * stream is playing, so these extra details may be provided in further
 * callbacks.
 *
 * The decoder may receive VNCCommonDecoderCallbackStreamPayload callbacks as
 * soon as this method has been returned from. The SDK need not wait for
 * VNCCommonDecoderSupportingAPIStreamCreateCompleted before starting to
 * forward payloads.
 *
 * \return None. VNCCommonDecoderSupportingAPIStreamError should be used to
 * notify the framework of errors.
 * \param pluginContext The context pointer returned by the entry point.
 * \param streamId The unique ID of this stream. IDs start from 1, and are
 * never re-used.
 * \param mediaType The type of media in the payloads – audio, video, or
 * multiplexed audio and video.
 * \param specificType The protocol-specific media type. See the constants for
 * more detail.
 */
typedef void VNCCALL
VNCCommonDecoderCallbackStreamCreated(
    VNCCommonDecoderPluginContext pluginContext,
    vnc_uint64_t streamId,
    VNCCommonDecoderMediaType mediaType,
    VNCCommonDecoderStreamSpecificType specificType);

/**
 * \brief May optionally be called to ask the decoder if it supports a
 * particular LPCM encoding.
 * \ingroup decodercallbacks
 *
 * This callback is optional, and need not be implemented by the decoder. If it
 * is not implemented, the framework will assume that all LPCM encodings are
 * unsupported by the decoder.
 *
 * The decoder can optionally provide details of a similar encoding which is
 * supported.
 *
 * It is anticipated that, if integrating with WiFi Display in future, this API
 * will be used in place of VNCWiFiDisplayDecoderAddAudioCodec, and that the SDK
 * will use it to check each relevant encoding (rather than requiring the
 * decoder to advertise support for each one).
 *
 * This callback may be invoked at any point.
 *
 * Note: The return value should not be used to indicate whether or not the
 * formats are supported. The out parameter out_isEncodingSupported should be
 * used for this.
 *
 * \return VNCCommonDecoderError. If this is set to anything other than
 * VNCCommonDecoderErrorNone, then all out parameters will be ignored.
 * \param pluginContext The context pointer returned by the entry point.
 * \param formatDetails Details about the LPCM encoding being queried. This
 * struct is owned by the framework, and should not be used after this function
 * has returned. A struct size is not provided, as the contents of this struct
 * must never change. 
 * \param out_isEncodingSupported Should be set to vnc_true by the decoder if
 * the encoding specified by formatDetails is supported.
 * \param out_suggestedFormatDetailsList May optionally be filled in by the
 * decoder to point to the first element of an array of supported formats which
 * are similar to formatDetails. This array is owned by the decoder, and must
 * remain valid until the next call to this callback, or until the decoder is
 * destroyed. If out_suggestedFormatDetailsCount is zero, then this pointer may
 * be set to NULL.
 * \param out_suggestedFormatDetailsCount The decoder must set this to the
 * number of elements in out_suggestedFormatDetailsList. If no suggested
 * formats were provided, then this must be set to zero.
 */
typedef VNCCommonDecoderError VNCCALL
VNCCommonDecoderCallbackQueryEncodingSupportAudioLPCM(
    VNCCommonDecoderPluginContext pluginContext,
    const VNCCommonDecoderAudioFormatDetailsLPCM* formatDetails,
    vnc_bool_t* out_isEncodingSupported,
    VNCCommonDecoderAudioFormatDetailsLPCM** out_suggestedFormatDetailsList,
    vnc_size_t* out_suggestedFormatDetailsCount);

/**
 * \brief May optionally be called to let the decoder know the content of the
 * audio in the specified stream.
 * \ingroup decodercallbacks
 *
 * This callback is optional, and need not be implemented by the decoder.
 *
 * This callback may be invoked at any point after the framework has called
 * VNCCommonDecoderCallbackStreamCreated, and before the framework has invoked
 * VNCCommonDecoderCallbackStreamDestroyed. The SDK need not wait for
 * VNCCommonDecoderSupportingAPIStreamCreateCompleted before triggering this
 * callback.
 *
 * If the decoder does not recognise the types provided, it should ignore them.
 *
 * If the specified stream is unknown, the decoder should ignore this callback,
 * but may log out an error message.
 *
 * \return None. VNCCommonDecoderSupportingAPIStreamError should be used to
 * notify the framework of errors.
 * \param pluginContext The context pointer returned by the entry point.
 * \param streamId The unique ID of the relevant stream.
 * \param audioStreamContentTypeList Pointer to the first element in a list
 * of audio content types offered by this stream. This list is owned by the
 * framework, and should not be used after this function has returned.
 * \param audioStreamContentTypeCount The number of elements in
 * audioStreamContentTypeList. If this is zero (and hence the list is empty),
 * the content type is unknown.
 */
typedef void VNCCALL
VNCCommonDecoderCallbackStreamAudioContentHint(
    VNCCommonDecoderPluginContext pluginContext,
    vnc_uint64_t streamId,
    const VNCCommonDecoderAudioStreamContentType* audioStreamContentTypeList,
    vnc_size_t audioStreamContentTypeCount);

/**
 * \brief May optionally be called to let the decoder know the encoding type of
 * the audio in the specified stream.
 * \ingroup decodercallbacks
 *
 * This callback is optional, and need not be implemented by the decoder.
 *
 * This callback may be invoked at any point after the framework has called
 * VNCCommonDecoderCallbackStreamCreated, and before the framework has invoked
 * VNCCommonDecoderCallbackStreamDestroyed. The SDK need not wait for
 * VNCCommonDecoderSupportingAPIStreamCreateCompleted before triggering this
 * callback.
 *
 * This should take precedence over the contents of any previous call to a
 * VNCCommonDecoderCallbackStreamAudioEncodingHint… function.
 *
 * If the specified stream is unknown, the decoder should ignore this callback,
 * but may log out an error message.
 *
 * If the encoding is unsupported, or does not match the
 * VNCCommonDecoderMediaType specified in
 * VNCCommonDecoderCallbackStreamCreated, the decoder may invoke
 * VNCCommonDecoderSupportingAPIStreamError and may ignore any further
 * stream payloads.
 *
 * \return None. VNCCommonDecoderSupportingAPIStreamError should be used
 * to notify the SDK of an error.
 * \param pluginContext The context pointer returned by the entry point.
 * \param streamId The unique ID of the relevant stream.
 * \param formatDetails Details about this LPCM stream. This struct is owned by
 * the framework, and should not be used after this function has returned. A
 * struct size is not provided, as the contents of this struct must never
 * change.
 */
typedef void VNCCALL
VNCCommonDecoderCallbackStreamAudioEncodingHintLPCM(
    VNCCommonDecoderPluginContext pluginContext,
    vnc_uint64_t streamId,
    const VNCCommonDecoderAudioFormatDetailsLPCM* formatDetails);

/**
 * \brief May optionally be called to ask the decoder if it supports a
 * particular set of H.264 attributes and video mode combinations.
 * \ingroup decodercallbacks
 *
 * In one call to this decoder callback, the SDK may query an
 * unlimited number of video modes for one given set of H.264 attributes.
 *
 * Additionally, the decoder can optionally provide details of an unlimited
 * number of similar video modes which are supported for the provided set of
 * H.264 attributes.
 *
 * This callback is optional, and need not be implemented by the decoder. If it
 * is not implemented, the framework will assume that all H.264 encodings are
 * unsupported by the decoder. In this case, the framework will respond to the
 * SDK’s query by returning VNCCommonDecoderErrorNone, and will say that none
 * of the encodings are supported, and no suggested alternative video modes
 * were provided.
 *
 * This callback may be invoked at any point.
 *
 * Note: The return value should not be used to indicate whether or not the
 * formats are supported. The out parameter out_isEncodingSupportedList should
 * be used for this.
 *
 * \return VNCCommonDecoderError. If this is set to anything other than
 * VNCCommonDecoderErrorNone, then all out parameters will be ignored.
 * \param pluginContext The context pointer returned by the entry point.
 * \param h264AttributesSize The size of the h264Attributes struct, for
 * backwards compatibility reasons.
 * \param h264Attributes Details about the H.264 encoding being queried. This
 * struct is owned by the framework, and should not be used after this function
 * has returned.
 * \param videoModeList A pointer to the first element of an array of video
 * modes being queried. This is owned by the framework, and should not be used
 * after this function has returned. A struct size parameter is not provided,
 * as this struct must never change.
 * \param videoModeCount The number of elements in videoModeList and
 * out_isEncodingSupportedList.
 * \param out_isEncodingSupportedList Each element
 * out_isEncodingSupportedList[i] should be set by the decoder to indicate
 * whether or not the corresponding video mode videoModeList[i] is supported,
 * for the given set of H.264 attributes h264Attributes. Should be set to
 * vnc_true if the video mode/H.264 attributes pair is supported. This is owned
 * by the framework, and is of size videoModeCount.
 * \param out_suggestedVideoModeList May optionally be filled in by the decoder
 * to point to the first element of an array of similar supported video modes
 * for the given set of H.264 attributes. This array is owned by the decoder,
 * and must remain valid until the next call to this callback, or until the
 * decoder is destroyed. If out_suggestedVideoModeCount is zero, then this
 * pointer may be set to NULL.
 * \param out_suggestedVideoModeCount The decoder must set this to the number
 * of elements in out_suggestedVideoModeList. If no suggested video modes were
 * provided, then this must be set to zero.
 */
typedef VNCCommonDecoderError VNCCALL
VNCCommonDecoderCallbackQueryEncodingSupportVideoH264(
    VNCCommonDecoderPluginContext pluginContext,
    vnc_size_t h264AttributesSize,
    const VNCCommonDecoderH264Attributes* h264Attributes,
    const VNCCommonDecoderVideoMode* videoModeList,
    vnc_size_t videoModeCount,
    vnc_bool_t* out_isEncodingSupportedList,
    VNCCommonDecoderVideoMode** out_suggestedVideoModeList,
    vnc_size_t* out_suggestedVideoModeCount);

/**
 * \brief May optionally be called to let the decoder know the encoding type of
 * the video in the specified stream.
 * \ingroup decodercallbacks
 *
 * This callback is optional, and need not be implemented by the decoder.
 *
 * This callback may be invoked at any point after the framework has called
 * VNCCommonDecoderCallbackStreamCreated, and before the framework has invoked
 * VNCCommonDecoderCallbackStreamDestroyed. The SDK need not wait for
 * VNCCommonDecoderSupportingAPIStreamCreateCompleted before triggering
 * this callback.
 *
 * This should take precedence over the contents of any previous call to a
 * VNCCommonDecoderCallbackStreamVideoEncodingHint… function.
 *
 * If the specified stream is unknown, the decoder should ignore this callback,
 * but may log out an error message.
 *
 * If the encoding is unsupported, or does not match the
 * VNCCommonDecoderMediaType specified in
 * VNCCommonDecoderCallbackStreamCreated, the decoder may invoke
 * VNCCommonDecoderSupportingAPIStreamError and may ignore any further
 * stream payloads. Alternatively, it could try and render the payloads anyway,
 * as the H.264 stream itself may contain metadata that differs from this hint.
 *
 * \return None. VNCCommonDecoderSupportingAPIStreamError should be used
 * to notify the SDK of an error.
 * \param pluginContext The context pointer returned by the entry point.
 * \param streamId The unique ID of the relevant stream.
 * \param h264AttributesSize The size of the h264Attributes struct, for
 * backwards compatibility reasons.
 * \param h264Attributes Details about the H.264 encoding which is now in use.
 * This struct is owned by the framework, and should not be used after this
 * function has returned.
 * \param videoMode Details about the video mode which is now in use. This may
 * be NULL if this is not known. This struct is owned by the framework, and
 * should not be used after this function has returned. A struct size parameter
 * is not provided, as this struct must never change.
 */
typedef void VNCCALL
VNCCommonDecoderCallbackStreamVideoEncodingHintH264(
    VNCCommonDecoderPluginContext pluginContext,
    vnc_uint64_t streamId,
    vnc_size_t h264AttributesSize,
    const VNCCommonDecoderH264Attributes* h264Attributes,
    const VNCCommonDecoderVideoMode* videoMode);

/**
 * \brief Informs the decoder that a stream has started playing. The decoder
 * should start rendering payloads received after this callback, and may choose
 * to ignore payloads received before it.
 * \ingroup decodercallbacks
 *
 * This callback is not optional, and must be implemented by the decoder.
 *
 * This is an asynchronous callback. The decoder should notify the framework
 * that the stream has been started using the
 * VNCCommonDecoderSupportingAPIStreamStartCompleted supporting API.
 *
 * The stream may be started and stopped multiple times.
 *
 * \return None. VNCCommonDecoderSupportingAPIStreamError should be used
 * to notify the SDK of an error.
 * \param pluginContext The context pointer returned by the entry point.
 * \param streamId The unique ID of the relevant stream.
 */
typedef void VNCCALL
VNCCommonDecoderCallbackStreamStarted(
    VNCCommonDecoderPluginContext pluginContext,
    vnc_uint64_t streamId);

/**
 * \brief Informs the decoder that a stream has stopped playing. The decoder
 * may choose to ignore payloads received after this callback.
 * \ingroup decodercallbacks
 *
 * This callback is not optional, and must be implemented by the decoder.
 *
 * This is an asynchronous callback. The decoder should notify the framework
 * that the stream has been stopped using the
 * VNCCommonDecoderSupportingAPIStreamStopCompleted supporting API.
 *
 * All payloads received before this callback must have been rendered by the
 * time VNCCommonDecoderSupportingAPIStreamStopCompleted is called, unless the
 * stopImmediately parameter is set.
 *
 * Each stream may be started and stopped multiple times.
 *
 * \return None. VNCCommonDecoderSupportingAPIStreamError should be used
 * to notify the SDK of an error.
 * \param pluginContext The context pointer returned by the entry point.
 * \param streamId The unique ID of the relevant stream.
 * \param stopImmediately If this is true, the decoder should stop the stream
 * without waiting for further payloads to render. If this is false, the
 * decoder should wait until all payloads (which were received before this
 * callback) have finished rendering.
 */
typedef void VNCCALL
VNCCommonDecoderCallbackStreamStopped(
    VNCCommonDecoderPluginContext pluginContext,
    vnc_uint64_t streamId,
    vnc_bool_t stopImmediately);

/**
 * \brief Informs the decoder that a stream no longer exists.
 * \ingroup decodercallbacks
 *
 * This callback is not optional, and must be implemented by the decoder.
 *
 * This is an asynchronous callback. The decoder should notify the framework
 * that the stream has been destroyed using the
 * VNCCommonDecoderSupportingAPIStreamDestroyCompleted supporting API.
 *
 * The decoder must have finished moving the stream to the "Stopped" state
 * before this is invoked.
 *
 * All payloads specific to this stream must have been returned to the
 * framework before VNCCommonDecoderSupportingAPIStreamDestroyCompleted is
 * called.
 *
 * \return None. VNCCommonDecoderSupportingAPIStreamError should be used
 * to notify the SDK of an error.
 * \param pluginContext The context pointer returned by the entry point.
 * \param streamId The unique ID of the relevant stream.
 */
typedef void VNCCALL
VNCCommonDecoderCallbackStreamDestroyed(
    VNCCommonDecoderPluginContext pluginContext,
    vnc_uint64_t streamId);

/**
 * \brief Informs the decoder that a payload is available.
 * \ingroup decodercallbacks
 *
 * This callback is not optional, and must be implemented by the decoder.
 *
 * This callback may be invoked from any thread, including the SDK thread.
 *
 * This callback is only invoked between calls to
 * VNCCommonDecoderCallbackStreamCreated and
 * VNCCommonDecoderCallbackStreamDestroyed.
 *
 * Payloads received when the stream is not in the "Playing" state should be
 * silently ignored.
 *
 * All payloads provided to the decoder should be returned to the framework
 * using VNCCommonDecoderSupportingAPIStreamRecyclePayload. No payloads (for
 * this particular stream) should be in the decoder’s possession by the time
 * VNCCommonDecoderSupportingAPIStreamDestroyCompleted is called.
 *
 * \return None. VNCCommonDecoderSupportingAPIStreamError or
 * VNCCommonDecoderSupportingAPIError should be used to notify the
 * framework of errors.
 * \param pluginContext The context pointer returned by the entry point.
 * \param streamId The unique ID of the relevant stream.
 * \param payloadStructSize The size of the payload struct, for backwards
 * compatibility reasons.
 * \param payload A pointer to the payload struct. Ownership of this is
 * transferred to the decoder, but all payloads must eventually be returned to
 * the framework.
 */
typedef void VNCCALL
VNCCommonDecoderCallbackStreamPayload(
    VNCCommonDecoderPluginContext pluginContext,
    vnc_uint64_t streamId,
    vnc_size_t payloadStructSize,
    VNCCommonDecoderStreamPayload* payload);

/**
 * \brief Instructs the decoder to mute the specified audio stream. The decoder
 * should continue to decode payloads, but should not render them.
 * \ingroup decodercallbacks
 *
 * This callback is not optional, and must be implemented by the decoder.
 *
 * This callback may be invoked at any time while the stream exists, whether or
 * not it is playing.
 *
 * If the stream is already muted, this callback should be ignored. If the
 * stream ID is unrecognised, or the specified stream does not contain audio,
 * the decoder should silently ignore the callback, but may log an error.
 *
 * Ducking and muting may occur at the same time. In this case, muting should
 * take precedence.
 *
 * Only the specified stream should be affected by this callback.
 *
 * Streams should start in an un-muted state.
 *
 * \return None. VNCCommonDecoderSupportingAPIStreamError or
 * VNCCommonDecoderSupportingAPIError should be used to notify the
 * framework of errors.
 * \param pluginContext The context pointer returned by the entry point.
 * \param streamId The unique ID of the relevant stream.
 */
typedef void VNCCALL
VNCCommonDecoderCallbackStreamMuteEnable(
    VNCCommonDecoderPluginContext pluginContext,
    vnc_uint64_t streamId);

/**
 * \brief Instructs the decoder to un-mute the specified audio stream. The
 * decoder should start rendering payloads again.
 * \ingroup decodercallbacks
 *
 * This callback is not optional, and must be implemented by the decoder.
 *
 * This callback may be invoked at any time while the stream exists, whether or
 * not it is playing.
 *
 * If the stream is already un-muted, this callback should be ignored. If the
 * stream ID is unrecognised, or the specified stream does not contain audio,
 * the decoder should silently ignore the callback, but may log an error.
 *
 * Ducking and muting may occur at the same time. In this case, muting should
 * take precedence.
 *
 * Only the specified stream should be affected by this callback.
 *
 * Streams should start in an un-muted state.
 *
 * \return None. VNCCommonDecoderSupportingAPIStreamError or
 * VNCCommonDecoderSupportingAPIError should be used to notify the
 * framework of errors.
 * \param pluginContext The context pointer returned by the entry point.
 * \param streamId The unique ID of the relevant stream.
 */
typedef void VNCCALL
VNCCommonDecoderCallbackStreamMuteDisable(
    VNCCommonDecoderPluginContext pluginContext,
    vnc_uint64_t streamId);

/**
 * \brief Instructs the decoder to duck the specified audio stream. The decoder
 * should gradually reduce the stream volume to a lower level.
 * \ingroup decodercallbacks
 *
 * This callback is optional, and need not be implemented by the decoder.
 * However, ducking support may be required for some protocols.
 *
 * This callback may be invoked at any time while the stream exists, whether or
 * not it is playing.
 *
 * If the stream is already ducked, this callback should be ignored. If the
 * stream ID is unrecognised, or the specified stream does not contain audio,
 * the decoder should silently ignore the callback, but may log an error.
 *
 * The decoder should correctly handle the case where
 * VNCCommonDecoderCallbackStreamDuckDisable is called before the volume ramp
 * is complete. Audio artifacts should not be present if this occurs.
 *
 * Ducking and muting may occur at the same time. In this case, muting should
 * take precedence.
 *
 * Only the specified stream should be affected by this callback.
 *
 * Streams should start in an un-ducked state.
 *
 * \return None. VNCCommonDecoderSupportingAPIStreamError or
 * VNCCommonDecoderSupportingAPIError should be used to notify the
 * framework of errors.
 * \param pluginContext The context pointer returned by the entry point.
 * \param streamId The unique ID of the relevant stream.
 * \param suggestedRampMs The suggested duration of the volume ramp down, in
 * milliseconds. The decoder may ignore this. If this is zero, then the server
 * has not specified a value, and a sensible default should be used (e.g.
 * 200ms).
 * \param suggestedAttenuationDb The suggested attenuation at the end of the
 * volume ramp, in decibels. The decoder may ignore this. If this is zero, then
 * the server has not specified a value, and a sensible default should be used.
 * This value is always a positive integer -- if the server sends a negative
 * value, this will be made positive.
 */
typedef void VNCCALL
VNCCommonDecoderCallbackStreamDuckEnable(
    VNCCommonDecoderPluginContext pluginContext,
    vnc_uint64_t streamId,
    vnc_uint64_t suggestedRampMs,
    vnc_uint32_t suggestedAttenuationDb);

/**
 * \brief Instructs the decoder to un-duck the specified audio stream. The
 * decoder should gradually increase the stream volume to its initial level.
 * \ingroup decodercallbacks
 *
 * This callback is optional, and need not be implemented by the decoder.
 * However, ducking support may be required for some protocols.
 *
 * This callback may be invoked at any time while the stream exists, whether or
 * not it is playing.
 *
 * If the stream is already un-ducked, this callback should be ignored. If the
 * stream ID is unrecognised, or the specified stream does not contain audio,
 * the decoder should silently ignore the callback, but may log an error.
 *
 * The decoder should correctly handle the case where
 * VNCCommonDecoderCallbackStreamDuckDisable is called before the volume ramp
 * is complete. Audio artifacts should be not be present if this occurs.
 *
 * The suggested ramp duration from VNCCommonDecoderCallbackStreamDuckEnable
 * may be used here if desired.
 *
 * Ducking and muting may occur at the same time. In this case, muting should
 * take precedence.
 *
 * Only the specified stream should be affected by this callback.
 *
 * Streams should start in an un-ducked state.
 *
 * \return None. VNCCommonDecoderSupportingAPIStreamError or
 * VNCCommonDecoderSupportingAPIError should be used to notify the
 * framework of errors.
 * \param pluginContext The context pointer returned by the entry point.
 * \param streamId The unique ID of the relevant stream.
 */
typedef void VNCCALL
VNCCommonDecoderCallbackStreamDuckDisable(
    VNCCommonDecoderPluginContext pluginContext,
    vnc_uint64_t streamId);

/**
 * \brief Invoked by the SDK if the media source has successfully
 * handled a call to VNCCommonDecoderSupportingAPIStreamRequestFrameRateLimit.
 * \ingroup decodercallbacks
 *
 * Depending on the behaviour of the media source, this callback may never be
 * invoked, or may be invoked multiple times.
 *
 * \param pluginContext The context pointer returned by the entry point.
 * \param streamId The unique ID of the relevant stream.
 * \param newFrameRate The new framerate adopted by the server. If this is
 * unknown, a value of zero is used.
 */
typedef void VNCCALL
VNCCommonDecoderCallbackStreamFrameRateLimitRequestSuccess(
    VNCCommonDecoderPluginContext pluginContext,
    vnc_uint64_t streamId,
    vnc_uint32_t newFrameRate);

/**
 * \brief Informs the decoder that the SDK user is attempting to set a decoder
 * property.
 * \ingroup decodercallbacks
 *
 * This callback is required, and must be implemented by the decoder.
 *
 * This callback may be invoked at any time while the decoder exists.
 *
 * The data type of the value is also provided, aiding type safety in the
 * BaseDecoder C++ wrapper.
 *
 * The SDK API for setting decoder properties will block until the decoder has
 * returned from this method.
 *
 * \return VNCCommonDecoderError. This will be passed unchanged from the
 * decoder to the SDK user.
 * \param pluginContext The context pointer returned by the entry point.
 * \param propertyKey The identifier of the property being set. This will be
 * passed unchanged from the SDK user to the decoder.
 * \param propertyType The data type pointed to by propertyValue. This will be
 * passed unchanged from the SDK user to the decoder.
 * \param propertyValue The new value of the property being set. This will be
 * passed unchanged from the SDK user to the decoder.
 */
typedef VNCCommonDecoderError VNCCALL
VNCCommonDecoderCallbackPropertySet(
    VNCCommonDecoderPluginContext pluginContext,
    VNCCommonDecoderPropertyKey propertyKey,
    VNCCommonDecoderDataType propertyType,
    void* propertyValue);

/**
 * \brief Triggered by the SDK user on every iteration of the SDK thread main
 * loop. This allows decoders to provide event handles to the SDK user, so that
 * the SDK user will notify them when activity occurs on those event handles.
 * \ingroup decodercallbacks
 *
 * Each decoder can provide zero or more handles for the SDK to monitor for it.
 * The SDK obtains these event handles by calling this method on every
 * iteration of its main loop. When activity is detected on these handles the
 * SDK will call VNCCommonDecoderCallbackEventHandleActivity(). This  prevents
 * the decoder from needing to run a separate thread for performing such
 * asynchronous actions.
 *
 * On Windows, VNCCommonDecoderEventHandle is a HANDLE that must identify  a
 * waitable kernel object. On UNIX platforms, VNCCommonDecoderEventHandle is a
 * file descriptor suitable for use in a select() call.
 *
 * On successful return, *out_eventHandleList should be set to point to an
 * array of handles and associated directions. The decoder retains ownership of
 * the array, and it should remain valid until the next call to
 * VNCCommonDecoderCallbackGetEventHandles() or
 * VNCCommonDecoderCallbackDestroy() for the decoder.
 *
 * Note that the SDK thread might not start until the first stream is created,
 * or in some cases may never start, meaning this callback will never be
 * invoked.
 *
 * This callback is required, and must be implemented by the decoder.
 *
 * \return None. VNCCommonDecoderSupportingAPIStreamError or
 * VNCCommonDecoderSupportingAPIError should be used to notify the
 * framework of errors.
 * \param pluginContext The context pointer returned by the entry point.
 * \param out_eventHandleList Should be set by the decoder to point to the
 * start of an array of the decoder’s event handles. This is owned by the
 * decoder, but must remain valid until the next call to
 * VNCCommonDecoderCallbackGetEventHandles, or until the decoder is destroyed.
 * \param out_eventHandleCount Should be set by the decoder to the number of
 * event handles in out_eventHandleList.
 */
typedef void VNCCALL
VNCCommonDecoderCallbackGetEventHandles(
    VNCCommonDecoderPluginContext pluginContext,
    VNCCommonDecoderEventHandleParams** out_eventHandleList,
    vnc_size_t* out_eventHandleCount);

/**
 * \brief Triggered by the SDK when there is activity on one of the
 * event handles provided to VNCCommonDecoderCallbackGetEventHandles.
 * \ingroup decodercallbacks
 *
 * This callback is required, and must be implemented by the decoder.
 *
 * When activity is detected in the appropriate direction on one or more of the
 * event handles provided by the previous call to
 * VNCCommonDecoderCallbackGetEventHandles() then the SDK will call this
 * method.
 *
 * The decoder should take care to clear any pending activity on the handles or
 * to not include them in the next call to
 * VNCCommonDecoderCallbackGetEventHandles(), otherwise the SDK will call this
 * method almost immediately again.
 *
 * \return None. VNCCommonDecoderSupportingAPIStreamError or
 * VNCCommonDecoderSupportingAPIError should be used to notify the
 * framework of errors.
 * \param pluginContext The context pointer returned by the entry point.
 * \param activeEventHandleList The event handles on which activity was
 * detected.
 * \param activeEventHandleCount The number of items in activeEventHandleList.
 * If this is zero, then the handles on which the activity occurred are
 * unspecified, and it is the responsibility of the decoder to determine which
 * handles are active.
 */
typedef void VNCCALL
VNCCommonDecoderCallbackEventHandleActivity(
    VNCCommonDecoderPluginContext pluginContext,
    const VNCCommonDecoderEventHandle* activeEventHandleList,
    vnc_size_t activeEventHandleCount);


/**
 * \brief Callbacks to be implemented by the decoder and called by the
 * framework.
 * \ingroup decodercallbacks
 */
typedef struct {

  /**
   * The last callback invoked on the decoder. No further decoder
   * callbacks should be received once this has returned.
   */
  VNCCommonDecoderCallbackDestroyed* vncCommonDecoderCallbackDestroyed;

  /**
   * Informs the decoder that a new stream is available.
   */
  VNCCommonDecoderCallbackStreamCreated* vncCommonDecoderCallbackStreamCreated;

  /**
   * May optionally be called to ask the decoder if it supports a
   * particular LPCM encoding.
   */
  VNCCommonDecoderCallbackQueryEncodingSupportAudioLPCM* vncCommonDecoderCallbackQueryEncodingSupportAudioLPCM;

  /**
   * May optionally be called to let the decoder know the content of the
   * audio in the specified stream.
   */
  VNCCommonDecoderCallbackStreamAudioContentHint* vncCommonDecoderCallbackStreamAudioContentHint;

  /**
   * May optionally be called to let the decoder know the encoding type of
   * the audio in the specified stream.
   */
  VNCCommonDecoderCallbackStreamAudioEncodingHintLPCM* vncCommonDecoderCallbackStreamAudioEncodingHintLPCM;

  /**
   * May optionally be called to ask the decoder if it supports a
   * particular set of H.264 attributes and video mode combinations.
   */
  VNCCommonDecoderCallbackQueryEncodingSupportVideoH264* vncCommonDecoderCallbackQueryEncodingSupportVideoH264;

  /**
   *  May optionally be called to let the decoder know the encoding type of
   * the video in the specified stream.
   */
  VNCCommonDecoderCallbackStreamVideoEncodingHintH264* vncCommonDecoderCallbackStreamVideoEncodingHintH264;

  /**
   * Informs the decoder that a stream has started playing.
   */
  VNCCommonDecoderCallbackStreamStarted* vncCommonDecoderCallbackStreamStarted;

  /**
   * Informs the decoder that a stream has stopped playing.
   */
  VNCCommonDecoderCallbackStreamStopped* vncCommonDecoderCallbackStreamStopped;

  /**
   * Informs the decoder that a stream no longer exists.
   */
  VNCCommonDecoderCallbackStreamDestroyed* vncCommonDecoderCallbackStreamDestroyed;

  /**
   * Informs the decoder that a payload is available.
   */
  VNCCommonDecoderCallbackStreamPayload* vncCommonDecoderCallbackStreamPayload;

  /**
   * Instructs the decoder to mute the specified audio stream.
   */
  VNCCommonDecoderCallbackStreamMuteEnable* vncCommonDecoderCallbackStreamMuteEnable;

  /**
   * Instructs the decoder to un-mute the specified audio stream.
   */
  VNCCommonDecoderCallbackStreamMuteDisable* vncCommonDecoderCallbackStreamMuteDisable;

  /**
   * Instructs the decoder to duck the specified audio stream.
   */
  VNCCommonDecoderCallbackStreamDuckEnable* vncCommonDecoderCallbackStreamDuckEnable;

  /**
   * Instructs the decoder to un-duck the specified audio stream.
   */
  VNCCommonDecoderCallbackStreamDuckDisable* vncCommonDecoderCallbackStreamDuckDisable;

  /**
   * Invoked by the SDK if the media source has successfully handled a call to
   * VNCCommonDecoderSupportingAPIStreamRequestFrameRateLimit
   */
  VNCCommonDecoderCallbackStreamFrameRateLimitRequestSuccess* vncCommonDecoderCallbackStreamFrameRateLimitRequestSuccess;

  /**
   * Informs the decoder that the SDK user is attempting to set a decoder
   * property.
   */
  VNCCommonDecoderCallbackPropertySet* vncCommonDecoderCallbackPropertySet;

  /**
   * Allows decoders to provide event handles to the SDK user, so that
   * the SDK user will notify them when activity occurs on those event handles.
   */
  VNCCommonDecoderCallbackGetEventHandles* vncCommonDecoderCallbackGetEventHandles;

  /**
   * Informs the decoder that a payload is available.
   */
  VNCCommonDecoderCallbackEventHandleActivity* vncCommonDecoderCallbackEventHandleActivity;

} VNCCommonDecoderCallbacks;


/**
 * \brief The type of the entry point to be exported by a decoder DLL
 * or shared object.
 *
 * The exported symbol name of the entry point should be
 * VNCCommonDecoderInitialize.
 *
 * The decoder must fill in the VNCCommonDecoderSupportInfo,
 * VNCCommonDecoderParameters, and VNCCommonDecoderCallbacks structs before
 * returning.
 *
 * The decoder is considered to be in the "Created" state by the time this has
 * returned, if and only if a non-NULL pointer is returned.
 *
 * If a NULL pointer is returned, no decoder callbacks will be invoked.
 *
 * \param decoderFrameworkContext This may be any value of the framework’s
 * choosing, and will be passed from the decoder to the framework in each
 * of the supporting APIs.
 *
 * \param applicationContext Opaque data type as provided by the
 * viewer application to the SDK when the decoder was registered or
 * loaded.
 *
 * \param out_decoderCallbacks Function pointers to each of the decoder
 * callbacks. The decoder must fill this in before returning from the entry
 * point. Note that the framework owns this struct, and it must not be used
 * after returning from the entry point.
 *
 * \param decoderCallbacksSize The size of *out_decoderCallbacks, for
 * backwards compatibility.
 *
 * \param decoderSupportingAPI Function pointers to each of the supporting
 * APIs, provided by the framework. Note that the framework owns this struct,
 * and it must not be used after returning from the entry point. Therefore,
 * the decoder should take a copy.
 *
 * \param decoderSupportingAPISize The size of *decoderSupportingAPI.
 *
 * \param sdkAttributes A struct of properties about the SDK. Note that the
 * framework owns this struct, and it must not be used after returning from the
 * entry point.
 *
 * \param sdkAttributesSize The size of *sdkAttributes.
 *
 * \param out_supportInfo Details about the decoder, to be filled in by the
 * decoder before this function returns. Any strings placed in the struct
 * should be allocated using VNCCommonDecoderSupportingAPIAllocString, and
 * ownership will transfer to the framework.
 *
 * \param supportInfoSize The size of the VNCCommonDecoderSupportInfo struct,
 * to allow for backwards compatibility.
 *
 * \param out_creationParameters Parameters which must be set by the decoder
 * before this function returns. Any strings placed in the struct should be
 * allocated using VNCCommonDecoderSupportingAPIAllocString, and ownership will
 * transfer to the framework.
 *
 * \param creationParametersSize The size of the VNCCommonDecoderCreationParams
 * struct, to allow for backwards compatibility.

 * \return Any value of the decoder’s choosing. This will be passed from the
 * framework to the decoder in each of the decoder callbacks.  The framework
 * will call VNCCommonDecoderCallbackDestroyed() when this object is no longer
 * required. Returning NULL here indicates that an error has occurred, and the
 * decoder will not be initialised.
 */
typedef VNCCommonDecoderPluginContext VNCCALL
VNCCommonDecoderInitializeType(
    VNCCommonDecoderFrameworkContext decoderFrameworkContext,
    VNCCommonDecoderApplicationContext applicationContext,
    VNCCommonDecoderCallbacks* out_decoderCallbacks,
    size_t decoderCallbacksSize,
    const VNCCommonDecoderSupportingAPI* decoderSupportingAPI,
    size_t decoderSupportingAPISize,
    const VNCCommonDecoderSDKAttributes* sdkAttributes,
    size_t sdkAttributesSize,
    VNCCommonDecoderSupportInfo* out_supportInfo,
    size_t supportInfoSize,
    VNCCommonDecoderCreationParams* out_creationParameters,
    size_t creationParametersSize);


#ifdef __cplusplus
}
#endif

#endif // VNCCOMMONDECODER_H_1665178272128288539943113463580786355

